<?php

namespace App\Http\Controllers;

use App\Models\Configuration;
use App\Models\Person;
use App\Models\Rol;
use App\Models\UsersPasswords;
use App\Models\UsersRol;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class UserController extends Controller
{
    /**
     * Start the login attempt
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        Artisan::call("cache:clear");
        // Receive the credentials
        $credentials = request(['email', 'password']);

        // Check if the email exists on the database
        if ($this->_checkEmail($credentials['email'])) {
            // Attempt to login with the credentials
            if (!$token = auth()->attempt($credentials, true)) {
                // Wrong login attempt, increment the login attempt
                User::where('email', $credentials['email'])->increment('wrong_login_attempt', 1, ['date_wrong_login_attempt' => Carbon::now()]);
                // Get the login attempts
                $attempts = User::select('email', 'wrong_login_attempt')->where('email', $credentials['email'])->get()->first();
                // Return fail login attempt and the number of attempts
                return response()->json(
                    [
                        'status' => 'fail',
                        'email' => $credentials['email'],
                        'attempts' => $attempts->wrong_login_attempt
                    ]
                    , 403
                );
            }
            // Updating the user data informacion
            $userData = $this->updatetokendata($token, $request->email, $request->ip());
            // Return the token with data
            return $this->respondWithToken($token);
        } else {
            return response()->json(
                [
                    'status' => 'fail',
                    'message' => 'email does not exists',
                ]
                , 403
            );
        }

    }

    /**
     * Register new user on the database
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function registernewuser(Request $request)
    {
        // Check if email exists
        if (!$this->_checkEmail($request->email)) {
            // Register the new user
            $query_response = $this->storagenewuser($request);
            // Return http response
            return response()->json(
                [
                    'status' => 'success',
                    'message' => $query_response
                ], 200
            );
        } else {
            // Return http response
            return response()->json(
                [
                    'status' => 'fail',
                    'message' => 'The email already exists!'
                ], 409
            );
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkemail(Request $request)
    {
        return response()->json(
            [
                'status'    =>  'success',
                'message'   =>  $this->_checkEmail($request->email)
            ], 200
        );
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function registerandlogin(Request $request)
    {
        if (!$this->_checkEmail($request->email)) {
            // Register the new user
            $this->storagenewuser($request);
            // Made the login
            $token = auth()->attempt(
                [
                    'email'     => $request->email,
                    'password'  => $request->password
                ]
            );
            // Update the token data
            $userData = $this->updatetokendata($token, $request->email, $request->ip());

            // Respond with the token
            return $this->respondWithToken($token);
        } else {
            // Return http response
            return response()->json(
                [
                    'status' => 'fail',
                    'message' => 'The email already exists!'
                ], 401
            );
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function registerexternaluser(Request $request)
    {
        $sid = Str::uuid()->toString();
        $configuration = Configuration::where('status', 'A')->get()->first();
        $person = Person::create(array(
            'users_sid'                 =>  $sid,
            'first_name'                =>  $request->first_name,
            'last_name'                 =>  $request->last_name,
            'phone_number'              =>  null,
            'intern'                    =>  0,
            'cargo_id'                  =>  $configuration->cargo_id,
            'area_id'                   =>  $configuration->area_id,
            'country_id'                =>  null,
            'status'                    =>  'A',
            'created_at'                =>  date("Y-m-d H:i:s")
        ));
        User::create(array(
            'sid'                       =>  $sid,
            'person_id'                 =>  $person->id,
            'global_branch_office_id'   =>  null,
            'profile_picture'           =>  null,
            'email'                     =>  $request->email,
            'password'                  =>  $request->password,
            'remember_token'            =>  null,
            'ttl'                       =>  43200,
            'wrong_login_attempt'       =>  0,
            'date_wrong_login_attempt'  =>  null,
            'client_ip'                 =>  $request->ip(),
            'code'                      =>  null,
            'status'                    =>  'A',
            'created_at'                =>  date("Y-m-d H:i:s"),
            'updated_at'                =>  null
        ));
        UsersPasswords::create(
            array(
                'users_sid'             => $sid,
                'users_id'              => $sid,
                'password'              => $request->encrypted,
                'status'                => 'A',
                'created_at'            => date("Y-m-d H:i:s")
            )
        );
        $rol = Rol::where('status', 'A')->where('rol_name', '*')->get()->first();
        UsersRol::create(array(
            'users_sid'                 =>  $sid,
            'global_rol_id'             =>  $rol->id
        ));
        $token = auth()->attempt(
            [
                'email'         => $request->email,
                'password'      => $request->password
            ]
        );
        // Update the token data
        $this->updatetokendata($token, $request->email, $request->ip());
        // Respond with the token
        return $this->respondWithToken($token);
    }

    /**
     * Verify it the token is valid
     * @return \Illuminate\Http\JsonResponse
     */
    public function checktoken()
    {
        // Check the token session
        $response = auth()->check();
        // If token valid
        if ($response) {
            // Get the user SID
            $user = $this->_getusersid();
            // Get the user information
            $userData = User::select('remember_token')->where('sid', $user)->get()->first();
            // Return the new token data
            return $this->respondWithToken($userData->remember_token);
        } else {
            return response()->json([
                'message' => 'Invalid token',
            ], 401);
        }
    }

    /**
     * Logout the user
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        // Obtenemos los datos del usuario actual
        $users_sid = $this->getusersid();
        // Con esta informacion procedemos a borrar su token
        User::where('sid', $users_sid)
            ->update(
                array(
                    'remember_token'    =>  NULL
                )
            );
        // Ahora cerramos la sesion
        auth()->logout();
        // Return http response
        return response()->json(
            [
                'status' => 'success',
            ], 200
        );
    }

    /**
     * Check if the email exists in the user table
     * @param $email
     * @return bool
     */
    private function _checkEmail($email)
    {
        return (User::where('email', $email)->exists()) ? true : false;
    }

    /**
     * @param $token
     * @param $email
     * @param $ip
     * @return mixed
     */
    private function updatetokendata($token, $email, $ip)
    {
        // Update the token data
        User::where('email', $email)
            ->update(
                [
                    'remember_token' => $token,
                    'wrong_login_attempt' => 0,
                    'client_ip' => $ip,
                    'ttl' => (auth()->factory()->getTTL() * 60)
                ]
            );
        // return the result
        return User::where('remember_token', $token)->get()->first();
    }

    /**
     * @param Request $request
     * @return mixed
     */
    private function storagenewuser(Request $request)
    {
        $sid = Str::uuid()->toString();
        $myuser = $this->_getusersid();
        $query_response = User::create(array(
            'sid' => $sid,
            'person_id' => $request->person_id,
            'global_branch_office_id' => null,
            'email' => $request->email,
            'password' => $request->password,
            'ttl' => 43200,
            'wrong_login_attempt' => 0,
            'date_wrong_login_attempt' => null,
            'client_ip' => $request->ip(),
            'code' => null,
            'status' => 'A',
            'created_at' => date("Y-m-d H:i:s"),
            'updated_at' => null
        ));
        UsersPasswords::create(
            array(
                'users_sid' => $myuser,
                'users_id' => $sid,
                'password' => $request->encrypted,
                'status' => 'A',
                'created_at' => date("Y-m-d H:i:s")
            )
        );
        return $query_response;
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getpasswordfromuser(Request $request)
    {
        $response = UsersPasswords::where('status', 'A')->where('users_id', $request->id)->get()->first();
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getuserbysid(Request $request)
    {
        $response = User::select('person_id', 'email')->where('sid', $request->sid)->get()->first();
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateuserinfo(Request $request)
    {
        if ($this->validemailforupdate($request->email, $request->sid)) {
            $response = User::where('sid', $request->sid)->where('status', 'A')
                ->update(
                    array(
                        'email' => $request->email,
                        'person_id' => $request->person_id,
                    )
                );
            return response()->json(
                [
                    'status' => 'success',
                    'message' => $response
                ], 200
            );
        } else {
            return response()->json(
                [
                    'status' => 'fail',
                    'message' => 'The email already exists!'
                ], 409
            );
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getuserbyid(Request $request)
    {
        $response = User::where('status', 'A')
            ->where('sid', $request->sid)
            ->with(
                [
                    'person'
                ]
            )->get()->first();
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updatepassword(Request $request)
    {
        $response = User::where('sid', $request->sid)->where('status', 'A')
            ->update(
                array(
                    'password' => bcrypt($request->password)
                )
            );
        UsersPasswords::where('users_id', $request->sid)->update(
            array(
                'password' => $request->encrypted,
                'updated_at' => date("Y-m-d H:i:s")
            )
        );
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @param $email
     * @param $sid
     * @return bool
     */
    private function validemailforupdate($email, $sid)
    {
        if (User::where('sid', $sid)->where('email', $email)->exists()) {
            return true;
        } else {
            return User::where('email', $email)->where('sid', '<>', $sid)->exists() ? false : true;
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteuser(Request $request)
    {
        $response = User::where('sid', $request->sid)->update(['status' => 'X']);
        $passwords = UsersPasswords::where('users_id', $request->sid)->update(['status' => 'X']);
        return response()->json(
            [
                'status'    =>  'success',
                'message'   =>  $response,
                'password'  =>  $passwords
            ], 200
        );
    }

    /**
     * @return \Illuminate\Http\JsonResponse
     */
    public function getmyownperson()
    {
        $users_sid = $this->_getusersid();
        $response = User::where('sid', $users_sid)
            ->with(
                [
                    'person'
                ]
            )->get()->first();
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getallusersperpage(Request $request)
    {
        //where('sid', '<>', '69abe0de-7812-41bd-9124-6a7238aee7d1')->
        $response = User::where('status', 'A')
        ->where('sid', '<>', '69abe0de-7812-41bd-9124-6a7238aee7d1')
        ->with(
            [
                'person', 'usersrol'
            ]
        )->orderBy('created_at', 'ASC')->paginate($request->pagination);
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchusersperpage(Request $request)
    {
        //where('sid', '<>', '69abe0de-7812-41bd-9124-6a7238aee7d1')
        $person = new Person();
        $response = User::where('status', 'A')
            ->where('sid', '<>', '69abe0de-7812-41bd-9124-6a7238aee7d1')
            ->whereIn('person_id', function($query)use($person, $request){
                $query->select('id')->from($person->getTableName())->where('status', 'A')
                ->where(DB::raw("CONCAT(`first_name`, ' ', `last_name`)"), 'LIKE', '%'.$request->search.'%');
            })
            ->orWhere('email', 'LIKE', '%' . $request->search . '%')
            ->where('status', 'A')
            ->with(['person', 'usersrol'])
            ->orderBy('created_at', 'ASC')
            ->paginate($request->pagination);
        return response()->json(
            [
                'status' => 'success',
                'message' => $response
            ], 200
        );
    }

    /**
     * @return mixed
     */
    private function _getusersid()
    {
        $user = response()->json(auth()->user());
        return $user->original->sid;
    }

    /**
     * Get the token array structure.
     *
     * @param string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token)
    {
        $user = User::where('remember_token', $token)->with('person')->get()->first();
        $fullname = $user['person']['first_name'] . " " . $user['person']['last_name'];
        $profile = ($user['profile_picture']!=null)?$user['profile_picture']:null;
        $usersRol = UsersRol::where('users_sid', $user['sid'])->with('rol')->get();
        foreach ($usersRol as $value)
        {
            $rol[] = $value['rol']['rol_name'];
        }
        // Just return the token and basic data
        return response()->json([
            'access_token'  =>  $token,
            base64_encode('rol') => base64_encode(base64_encode(json_encode($rol))),
            'fullname'      =>  $fullname,
            'profile'       =>  ($profile!=null)?base64_encode($profile):null
        ], 200);
    }

    /**
     * @return mixed
     */
    private function getusersid()
    {
        $user = response()->json(auth()->user());
        return $user->original->sid;
    }

}
